/*
Small Reversi implementation by Jakub Debski '2011
It would be much more readable with structs but CC65 generates too unefficient code with them

We do not use potential_mobility_weight - the influence is too small comparing to time needed to compute
*/
#define SHOW_DEBUG 0
#define SHOW_HASH_TABLE_FULL 0

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "gui.h"

#define __fastcall__

#ifndef __ATARI__
	#define nop() __asm nop;
#else
	#define nop()
#endif

#define MAX_TURNS 60

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

void copy_new_board_to_visible();
void show_board();
void show_score();
void configure_game();

#ifndef __ATARI__
#define print_value(x,y,val)
#define print_string(x,y,s)
#define wait_for_fire()
#else
void print_value(unsigned char x,unsigned char y,unsigned char val);
void print_string(unsigned char x,unsigned char y,char *s);
void wait_for_fire();
#endif

//////////////////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////////////////

byte current_color=BLACK;
bool black_computer_controlled=false;
bool white_computer_controlled=true;

byte move_x, move_y;
byte best_x, best_y; // best move AI calculated
byte last_x, last_y, last_color;

byte game_turn; // game turn
bool ai_time_passed;
byte tree_depth=2; // must be at least 1, 3 is fine with alpha_beta and 1024 hashed boards
byte last_board_turn; // to speed up

//////////////////////////////////////////////////////////////////////////
// Local variables
//////////////////////////////////////////////////////////////////////////

byte local_x;
byte local_y;
index_type local_index;
byte local_temp1;
byte local_temp2;

//////////////////////////////////////////////////////////////////////////
// Data used currently - not encoded
//////////////////////////////////////////////////////////////////////////


// current board as current human player see it
byte visible_board[8][8];
byte visible_board_white_pieces;
byte visible_board_black_pieces;

// current boards that AI processes
size_t new_board_index=0;
byte new_board[8][8];
byte new_board_hash;
byte new_board_white_pieces;
byte new_board_black_pieces;
bool new_board_possible_moves[8][8]; 
byte new_board_possible_moves_number;
//signed int new_board_potential_mobility;

size_t old_board_index;
byte old_board[8][8];

//////////////////////////////////////////////////////////////////////////
// AI weights
//////////////////////////////////////////////////////////////////////////

#if !__ATARI__
signed int white_weights[GAME_PARTS][e_max_weight];
signed int black_weights[GAME_PARTS][e_max_weight];
#endif


//////////////////////////////////////////////////////////////////////////
// Encoded boards
//////////////////////////////////////////////////////////////////////////

// we need 2 bits to encode (none, white, black, possible move) so each line (8 cells) is encoded in word
word boards[MAX_BOARDS+1][8];
byte boards_hash[MAX_BOARDS+1];
byte boards_turn[MAX_BOARDS+1]; // if board turn is 0 then it's free to use
byte boards_possible_moves_count[MAX_BOARDS+1];
//signed int boards_potential_mobility[MAX_BOARDS+1];
size_t boards_parent[MAX_BOARDS+1]; // index of the parent board to this one (parent boards)
byte boards_white_pieces[MAX_BOARDS+1];
byte boards_black_pieces[MAX_BOARDS+1];

void __fastcall__ encode_new_board(size_t board_index)
{
	word e;
	for (local_y=0;local_y<8;++local_y)
	{
		e=0;
		for (local_x=0;local_x<8;++local_x)
		{
			e |= new_board[local_y][local_x] << (local_x*2);
			if (new_board_possible_moves[local_y][local_x])
				e |=  POSSIBLE_MOVE << (local_x*2);
		}
		boards[board_index][local_y]=e;
	}
	boards_white_pieces[board_index]=new_board_white_pieces;
	boards_black_pieces[board_index]=new_board_black_pieces;
	boards_possible_moves_count[board_index]=new_board_possible_moves_number;
	boards_hash[board_index]=new_board_hash;
	boards_turn[board_index]=game_turn;
//	boards_potential_mobility[board_index]=new_board_potential_mobility;
}

void decode_new_board(size_t board_index)
{
	new_board_possible_moves_number=0;
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			local_temp1=(boards[board_index][local_y] >> (local_x*2)) & 3;
			if (local_temp1==POSSIBLE_MOVE)
			{
				local_temp1=EMPTY;
				new_board_possible_moves[local_y][local_x]=true;
				++new_board_possible_moves_number;
			}
			else
				new_board_possible_moves[local_y][local_x]=false;

			new_board[local_y][local_x]=local_temp1;
		}
	}
	new_board_white_pieces=boards_white_pieces[board_index];
	new_board_black_pieces=boards_black_pieces[board_index];
	new_board_possible_moves_number=boards_possible_moves_count[board_index];
//	new_board_potential_mobility=boards_potential_mobility[board_index];
}

// Evaluates current ai_board
signed int piece_differential_weight;
signed int mobility_weight;
signed int edges_weight;
signed int corners_weight;
signed int x_weight;
signed int c_weight;
//signed int potential_mobility_weight;
signed int stable_disc_weight;

byte cornerx, cornery;
signed char deltax, deltay;
signed int white_score;
signed int black_score;
signed int total_white_score;
signed int total_black_score;
signed int total_score;

byte stable[3];

void check_corner()
{
	local_temp1=new_board[cornery][cornerx];
	if (stable_disc_weight!=0 && local_temp1!=EMPTY)
	{
		stable[WHITE]=0;
		stable[BLACK]=0;
		local_x=cornerx;
		for(;;)
		{
			local_x+=deltax;
			if (local_x>=8)
				break;
			if (new_board[cornery][local_x]==local_temp1)
				++stable[local_temp1];
			else
				break;
		};	
		local_y=cornery;
		for(;;)
		{
			local_y+=deltay;
			if (local_y>=8)
				break;
			if (new_board[local_y][cornerx]==local_temp1)
				++stable[local_temp1];
			else
				break;
		};	
		total_white_score+=stable_disc_weight*stable[WHITE];
		total_black_score+=stable_disc_weight*stable[BLACK];
	}

	// *** x-square differential (only consider new pieces, not flipped ones; only consider x-squares next to open corners)
	if (old_board[cornery][cornerx]==EMPTY)
	{
		// corners
		switch (local_temp1) // taken from stable
		{
			case WHITE:
				total_white_score+=corners_weight;
				break;
			case BLACK:
				total_black_score+=corners_weight;
				break;
		}

		// c-squares
		local_temp1=new_board[cornery][cornerx+deltax];
		if (local_temp1==WHITE)
			total_white_score+=c_weight;
		else if (local_temp1==BLACK)
			total_black_score+=c_weight;

		local_temp1=new_board[cornery+deltay][cornerx];
		if (local_temp1==WHITE)
			total_white_score+=c_weight;
		else if (local_temp1==BLACK)
			total_black_score+=c_weight;

		// x-square
		local_temp1=new_board[cornery+deltay][cornerx+deltax];
		if (local_temp1==WHITE)
			total_white_score+=x_weight;
		else if (local_temp1==BLACK)
			total_black_score+=x_weight;
	}
}

/*
void calculate_potential_mobility()
{
	byte opponent_color = (current_color == WHITE)? BLACK : WHITE; 
	signed char a,b;
	new_board_potential_mobility=0;
	for (local_y=1;local_y<7;++local_y)
	{
		for (local_x=1;local_x<7;++local_x)
		{
			local_temp2=new_board[local_y][local_x];
			if (local_temp2!=EMPTY)
			{
				for (b=-1;b<2;++b)
				{
					for (a=-1;a<2;++a)
					{
						if (local_temp1==EMPTY)
						{
							if (local_temp2==current_color)
								--new_board_potential_mobility;
							else
								++new_board_potential_mobility;
						}
					}
				}
			}
		}
	}
#if SHOW_DEBUG
	mvprintw(20,0,"Potential mobility %d  ",new_board_potential_mobility);
	refresh();
#endif
	return;
}
*/

// old_board_index must be set
int evaluate_new_board(size_t parent_board_index)
{
#if !__ATARI__
	signed int (*current_weights)[GAME_PARTS][e_max_weight];
	byte game_part;
#endif

	/*
	Evaluation is similar to http://lurgee.net/reversi-code/ which is quite reasonable.
	# Piece differential - a count of the number of pieces the player has gained minus the same count for the opponent; only pieces that have changed are considered.
	# Mobility - a count of the number of valid moves the player has minus the same count for the opponent; all pieces on the board are considered.
	#!Potential mobility - a count of the number of potential moves (open squares that are adjacent to an opponent's piece) that the player has minus the same count for the opponent; all pieces on the board are considered.
	# Corners - a count of the number of corners the player has gained minus the same count for the opponent; only pieces that have changed are considered.
	# Edges - a count of the number of A-squares and B-squares the player has gained minus the same count for the opponent; only pieces that have changed are considered.
	# X-squares - a count of the number of X-squares adjacent to open corners the player has gained minus the same count for the opponent; only placed pieces (not flipped pieces) are considered.
	# Wipe-out prevention - a move that results in a wipe-out for the player (a player losing all their pieces) is given a score of -1000.
	# Stage of the game - different weights are applied to each of the calculated scores for the various strategies. As indicated in the table below, these weights are changed depending on the stage of the game, which is represented by the number of pieces on the board. Some strategies are not relevant in certain stages of the game - the weight is set to 0 at these times. 
	*/
	white_score=0;
	black_score=0;
	total_score=0;
	total_white_score=0;
	total_black_score=0;

#if __ATARI__
	// Those weights are chosen by genetic algorithm and then a bit tuned by me
	if (last_board_turn!=game_turn)
	{
		last_board_turn=game_turn;
		if (game_turn<=16)
		{
			piece_differential_weight=-23;
			mobility_weight=1;
			edges_weight=-5;
			corners_weight=36;
			x_weight=-30;
			c_weight=-9;
//			potential_mobility_weight=-1;
			stable_disc_weight=10;
		}
		else if (game_turn<=32)
		{
			piece_differential_weight=-1;
			mobility_weight=1;
			edges_weight=-11;
			corners_weight=40;
			x_weight=-25;
			c_weight=-9;
//			potential_mobility_weight=-1;
			stable_disc_weight=18;
		}
		else if (game_turn<=49)
		{
			piece_differential_weight=-2;
			mobility_weight=3;
			edges_weight=1;
			corners_weight=58;
			x_weight=-30;
			c_weight=-19;
//			potential_mobility_weight=0;
			stable_disc_weight=11;
		}
		else if (game_turn<MAX_TURNS-tree_depth)
		{
			piece_differential_weight=-2;
			mobility_weight=17;
			edges_weight=11;
			corners_weight=26;
			x_weight=-11;
			c_weight=-11;
//			potential_mobility_weight=0;
			stable_disc_weight=21;
		}
		else
		{
			piece_differential_weight=1;
			mobility_weight=0;
			edges_weight=0;
			corners_weight=0;
			x_weight=0;
			c_weight=0;
//			potential_mobility_weight=0;
			stable_disc_weight=0;
		}
	}
#else
	if (game_turn<=16)
	{
		game_part=0;
	}
	else if (game_turn<=32)
	{
		game_part=1;
	}
	else if (game_turn<=49)
	{
		game_part=2;
	}
	else if (game_turn<MAX_TURNS-tree_depth)
	{
		game_part=3;
	}
	else
	{
		game_part=4;
	}
	
	if (current_color==WHITE)
		current_weights=&white_weights;
	else
		current_weights=&black_weights;

	piece_differential_weight=(*current_weights)[game_part][e_piece_differential_weight];
	mobility_weight=(*current_weights)[game_part][e_mobility_weight];
	edges_weight=(*current_weights)[game_part][e_edges_weight];
	corners_weight=(*current_weights)[game_part][e_corners_weight];
	x_weight=(*current_weights)[game_part][e_x_weight];
	c_weight=(*current_weights)[game_part][e_c_weight];
//	potential_mobility_weight=(*current_weights)[game_part][e_potential_mobility_weight];
	stable_disc_weight=(*current_weights)[game_part][e_stable_disc_weight];
#endif

	// *** piece change differential
	if (piece_differential_weight!= 0) 
	{		
		white_score = new_board_white_pieces;
		black_score = new_board_black_pieces;

		white_score -= boards_white_pieces[old_board_index];
		black_score -= boards_black_pieces[old_board_index];

		total_white_score = piece_differential_weight * white_score;
		total_black_score = piece_differential_weight * black_score;
	}

	// *** edge differential, excluding corners and c-squares (only consider changed pieces)
	if (edges_weight != 0) 
	{
		for ( local_y = 0; local_y < 8; local_y += 7) 
		{
				for (local_x = 1; local_x < 7; ++local_x) 
				{
					local_temp1 = new_board[local_y][local_x];
					if (old_board[local_y][local_x]!= local_temp1) 
					{
						switch (local_temp1)
						{
						case WHITE:
							total_white_score+=edges_weight;
							break;
						case BLACK:
							total_black_score+=edges_weight;
							break;
						}
					}
				}
		}
		for (local_y = 1; local_y < 7; ++local_y) 
		{
			for (local_x = 0; local_x < 8; local_x += 7) 
			{
				local_temp1 = new_board[local_y][local_x];
				if (old_board[local_y][local_x]!= local_temp1) 
				{
					switch (local_temp1)
					{
					case WHITE:
						total_white_score+=edges_weight;
						break;
					case BLACK:
						total_black_score+=edges_weight;
						break;
					}
				}
			}
		}
	}

	// check corner areas
	if (x_weight!=0 || corners_weight!=0 || c_weight!=0) 
	{
		cornerx=0,cornery=0,deltax=1,deltay=1;
		check_corner();
		cornerx=7,cornery=0,deltax=-1,deltay=1;
		check_corner();
		cornerx=0,cornery=7,deltax=1,deltay=-1;
		check_corner();
		cornerx=7,cornery=7,deltax=-1,deltay=-1;
		check_corner();
	}
	
	if (current_color==WHITE)
		total_score=total_white_score-total_black_score;
	else
		total_score=total_black_score-total_white_score;

	// *** mobility differential of current turn minus previous turn (other player)
	if (mobility_weight != 0) 
	{
		total_score += mobility_weight * ((signed int) new_board_possible_moves_number - (signed int) boards_possible_moves_count[old_board_index] - (signed int) boards_possible_moves_count[parent_board_index]);
	}

/*
	if (potential_mobility_weight!=0)
	{
//		total_score += potential_mobility_weight * (new_board_potential_mobility - boards_potential_mobility[old_board_index] - boards_potential_mobility[parent_board_index]);
		total_score += potential_mobility_weight * (new_board_potential_mobility);
	}
*/
	return total_score;
}

byte find_possible_moves_and_count_pieces()
{
	byte opponent_color = (current_color == WHITE)? BLACK : WHITE;  /* Identify opponent */

	signed char rowdelta;
	signed char coldelta;

	signed char row = 0;          /* Row index                        */
	signed char col = 0;          /* Column index                     */
	byte no_of_moves = 0;  /* Number of valid moves            */
	/* Initialize moves array to zero */
	memset(new_board_possible_moves,0,sizeof(new_board_possible_moves));
	new_board_white_pieces=0;
	new_board_black_pieces=0;

	/* Find squares for valid moves.                           */
	/* A valid move must be on a blank square and must enclose */
	/* at least one opponent square between two player squares */
	for(row = 0; row < 8; ++row)
	{
		for(col = 0; col < 8; ++col)
		{
			local_temp1=new_board[row][col];

			/* If not blank, then calculate pieces and continue  */
			if (local_temp1==WHITE)
			{
				++new_board_white_pieces;
				continue;
			}
			else if (local_temp1==BLACK)
			{
				++new_board_black_pieces;
				continue;
			}

			/* Check all the squares around the blank square  */ 
			/* for the opponents counter                      */
			for(rowdelta = -1; rowdelta <= 1; ++rowdelta)
			{
				for(coldelta = -1; coldelta <= 1; ++coldelta)
				{ 
					/* Don't check outside the array, or the current square */
					if(row + rowdelta < 0 || row + rowdelta >= 8 ||
						col + coldelta < 0 || col + coldelta >= 8 || 
						(rowdelta==0 && coldelta==0))
						continue;

					/* Now check the square */
					if(new_board[row + rowdelta][col + coldelta] == opponent_color)
					{
						/* If we find the opponent, move in the delta direction  */
						/* over opponent counters searching for a player counter */
						local_x = row + rowdelta;                /* Move to          */
						local_y = col + coldelta;                /* opponent square  */

						/* Look for a player square in the delta direction */
						for(;;)
						{
							local_x += rowdelta;                  /* Go to next square */
							local_y += coldelta;                  /* in delta direction*/

							/* If we move outside the array, give up */
							if(local_x >= 8 || local_y >= 8)
								break;

							/* If we find a blank square, give up */ 
							if(new_board[local_x][local_y] == EMPTY)
								break;
							/*  If the square has a player counter */
							/*  then we have a valid move          */
							if(new_board[local_x][local_y] == current_color)
							{
								new_board_possible_moves[row][col]=true;   /* Mark as valid */
								no_of_moves++;         /* Increase valid moves count */
								break;                 /* Go check another square    */
							}
						} 
					} 
				}  
			}
		}
	}
	return new_board_possible_moves_number=no_of_moves; 
}

void make_move(byte player_color)
{
	signed char rowdelta = 0;                   /* Row increment              */
	signed char coldelta = 0;                   /* Column increment           */
	byte opponent_color = (player_color == WHITE)? BLACK : WHITE;  /* Identify opponent */

	// 1. Set tile at the position

	new_board[move_y][move_x] = player_color;           /* Place the player counter   */

	// 2. Rotate tiles in all the directions.

	local_x = 0;                          
	local_y = 0;  
                       
	/* Check all the squares around this square */
	/* for the opponents counter                */
	for(rowdelta = -1; rowdelta <= 1; ++rowdelta)
	{
		for(coldelta = -1; coldelta <= 1; ++coldelta)
		{ 
			/* Don't check off the board, or the current square */
			if(move_y + rowdelta >= 8 || move_x + coldelta >= 8 || 
				(rowdelta==0 && coldelta== 0))
				continue;

			/* Now check the square */
			if(new_board[move_y + rowdelta][move_x + coldelta] == opponent_color)
			{
				/* If we find the opponent, search in the same direction */
				/* for a player counter                                  */
				local_x = move_y + rowdelta;        /* Move to opponent */
				local_y = move_x + coldelta;        /* square           */

				for(;;)
				{
					local_x += rowdelta;           /* Move to the      */
					local_y += coldelta;           /* next square      */ 

					/* If we are off the board give up */
					if(local_x >= 8 || local_y >= 8)
						break;

					/* If the square is blank give up */
					if(new_board[local_x][local_y] == EMPTY)
						break;

					/* If we find the player counter, go backwards from here */
					/* changing all the opponents counters to player         */
					if(new_board[local_x][local_y] == player_color)
					{
						while(new_board[local_x-=rowdelta][local_y-=coldelta]==opponent_color) /* Opponent? */
							new_board[local_x][local_y] = player_color;    /* Yes, change it */
						break;                     /* We are done    */
					} 
				}
			}
		}
	}
}

void calculate_new_board_hash()
{
	new_board_hash=0;
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			new_board_hash^=new_board[local_y][local_x]<<(local_x);
			new_board_hash+=local_y;
		}
	}
}

bool compare_new_board_with_encoded(size_t board_index)
{
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			local_temp1=(boards[board_index][local_y] >> (local_x*2)) & 3;
			if (local_temp1==POSSIBLE_MOVE)
				local_temp1=EMPTY;
			if (new_board[local_y][local_x]!=local_temp1)
				return false;
		}
	}
	return true;
}

size_t find_encoded_new_board_by_hash()
{
	for (local_index=0;local_index<MAX_BOARDS;++local_index)
	{
		if (boards_turn[local_index]!=0 && boards_hash[local_index]==new_board_hash)
		{
			// possibly the same?
			if (compare_new_board_with_encoded(local_index))
				return local_index;
		}
	}
	return INDEX_ERROR;
}

// Invalidate all boards that did not come from this move
void free_unnecessary_boards()
{
	size_t parent_index;
	size_t index;

	calculate_new_board_hash();
	new_board_index=find_encoded_new_board_by_hash();
	if (new_board_index==INDEX_ERROR) // this board is unknown so all other slots are invalid
	{
		// invalidate all boards
		memset(boards_turn,0,sizeof(boards_turn));
		old_board_index=INDEX_ERROR;
	}
	else // it is in some slot
	{
		old_board_index=new_board_index;
		boards_parent[new_board_index]=INDEX_ERROR; // this board is the first one, so does not have parent

		for (index=0;index<MAX_BOARDS;++index)
		{
			parent_index=index;
			for(;;)
			{
				if (boards_turn[parent_index]>=game_turn) // valid board
				{
					if (parent_index==new_board_index || boards_parent[parent_index]==new_board_index)
						break; // this board comes from the current one, don't invalidate it
					if (boards_parent[parent_index]==INDEX_ERROR)
					{
						boards_turn[index]=0; // invalidate this board, because some it's parent does not start from the current board
						break;
					}
					parent_index=boards_parent[parent_index];
					if (parent_index==INDEX_ERROR)
						break;
				}
				else
				{
					boards_turn[index]=0;
					break;
				}
			}
		}
	}
	return;
}

bool __fastcall__ check_zero_pieces_win()
{
	if (current_color==WHITE)
	{
		if (new_board_black_pieces==0)
			return true;
	}
	else
	{
		if (new_board_white_pieces==0)
			return true;
	}
	return false;
}

size_t find_free_board_slot()
{
	for (local_index=0;local_index<MAX_BOARDS;++local_index)
	{
		if (boards_turn[local_index]==0)
			return local_index;
	}
	return INDEX_ERROR;
}

byte create_move_list(byte *move_list)
{
	local_temp1=0;
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			if (new_board_possible_moves[local_y][local_x])
			{
				move_list[local_temp1]=local_x | local_y<<3;
				++local_temp1;
			}
		}
	}
	// randomize list
	for (local_x=0;local_x<local_temp1;++local_x)
	{
		local_y=my_random(local_temp1);
		local_temp2=move_list[local_x];
		move_list[local_x]=move_list[local_y];
		move_list[local_y]=local_temp2;
	}
	return local_temp1;
}

void __fastcall__ decode_move(byte encoded_move)
{
	move_x=encoded_move&7;
	move_y=encoded_move>>3;
}

byte __fastcall__ encode_move()
{
	return move_x | move_y<<3;
}

void switch_sides()
{
	if (current_color==BLACK)
		current_color=WHITE;
	else
		current_color=BLACK;
}

void next_turn()
{
	++game_turn;
	switch_sides();
}

void previous_turn()
{
	--game_turn;
	switch_sides();
}


#define max(a,b) ((a)>(b)?(a):(b))

// new_board - temporary board where everything happens

signed int alpha_beta(size_t parent_board_index, byte depth, int alpha, int beta)
{
	size_t board_index;
	byte move_list[MAXMOVES];
	byte move_i,total_moves;
	signed int value,localalpha=alpha,bestvalue=-INFINITY;

	calculate_new_board_hash();
	board_index = find_encoded_new_board_by_hash(); 
	if (board_index!=INDEX_ERROR) 
	{
		decode_new_board(board_index);
		if (new_board_possible_moves_number==0)
		{
			find_possible_moves_and_count_pieces();
//			calculate_potential_mobility();
		}
	}
	else // try to encode this board
	{
		find_possible_moves_and_count_pieces();
//		calculate_potential_mobility();

		if (depth!=0) // do not store leafs - there are too many
		{
			board_index = find_free_board_slot();
			if (board_index!=INDEX_ERROR)
			{
				encode_new_board(board_index);
				boards_parent[board_index]=parent_board_index;
			}
			else
			{
				// we have no slot to store this board, decrease alpha_beta_depth to evaluate this board
#if SHOW_HASH_TABLE_FULL
				mvaddstr(5,21,"Hash table full");
				refresh();
#endif
				//				depth=0;
				return -INFINITY;
			}		}
	}

#if SHOW_DEBUG
	copy_new_board_to_visible(); // !!!
	show_board();  // !!!
	mark_last_move();  // !!!
#endif

	if(check_zero_pieces_win()) 
	{
		return INFINITY; // ???
	}

	if(depth==0)	
	{
		if (old_board_index==INDEX_ERROR)
			old_board_index=parent_board_index;		
		return evaluate_new_board(parent_board_index);
	}

	assert(board_index!=INDEX_ERROR);
	if (board_index==INDEX_ERROR)
		return 0;

	total_moves = create_move_list(move_list);
	if(total_moves==0) 
		return -INFINITY;

	if (game_turn<4 || (depth==tree_depth && best_x==0xFF))
	{
		decode_move(move_list[my_random(total_moves)]);
		best_x=move_x;
		best_y=move_y;
	}
	
	if (game_turn<4) // make random move
		return INFINITY;

	// Simulate next turn
	for(move_i=0;move_i<total_moves;++move_i)
	{
		decode_move(move_list[move_i]);
		make_move(current_color);

		next_turn();

		value = -alpha_beta(board_index,depth-1,-beta,-localalpha);
		previous_turn();

		// restore board
		decode_new_board(board_index);

#if SHOW_DEBUG
		copy_new_board_to_visible(); // !!!
		show_board();                // !!!
#endif
	
		if (game_turn<MAX_TURNS-tree_depth) // use alpha-beta
		{
			bestvalue = max(value,bestvalue);
			if (bestvalue==INFINITY && depth==tree_depth)
			{
				decode_move(move_list[move_i]);
				best_x=move_x;
				best_y=move_y;
			}
			if(bestvalue >= beta) 
				break;
			if(bestvalue>localalpha) // important >, not >=, because may happen many times
			{
#if SHOW_DEBUG
				mvprintw(15,0,"BEST VALUE: %d  ",bestvalue);
				refresh();
				if (bestvalue<0)
					print_string(0,20,"BEST -  ");
				else
					print_string(0,20,"BEST    ");
				print_value(6,20,bestvalue);
#endif
				localalpha = bestvalue;
				if (depth==tree_depth)
				{
					decode_move(move_list[move_i]);
					best_x=move_x;
					best_y=move_y;
				}
			}
		}
		else // use negamax
		{
			if(value>bestvalue)
			{
				bestvalue=value;

#if SHOW_DEBUG
				mvprintw(15,0,"BEST VALUE: %d  ",bestvalue);
				refresh();
#endif
				if (depth==tree_depth)
				{
					decode_move(move_list[move_i]);
					best_x=move_x;
					best_y=move_y;
				}
			}
		}
	}
	return bestvalue;
}

byte get_ai_action()
{
	// 1. Free unnecessary boards (that are not delivered from this new_board)
	free_unnecessary_boards();

	// copy new_board to old_board for evaluation
	memcpy(old_board,new_board,sizeof(old_board));

	// 2. Find the best move
	best_x=0xFF;
	alpha_beta(-1,tree_depth,-INFINITY,INFINITY); // change to alpha_beta_first that checks move
	move_x=best_x;
	move_y=best_y;
	if (best_x==0xFF)
		return false;
	return true;
}


void init_first_board()
{
	// clear the board and set the starting positions
	memset(new_board,EMPTY,sizeof(new_board));
	new_board[3][3]=WHITE;
	new_board[3][4]=BLACK;
	new_board[4][3]=BLACK;
	new_board[4][4]=WHITE;
	last_x=3;
	last_y=2;
	new_board_index=0;
	old_board_index=-1;
	last_board_turn=0;
	game_turn=1;
	// invalidate all boards
	memset(boards_turn,0,sizeof(boards_turn));
}

void copy_new_board_to_visible()
{
#if !GENETIC_MODE
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			if (new_board_possible_moves[local_y][local_x])
				local_temp1=POSSIBLE_MOVE;
			else
				local_temp1=new_board[local_y][local_x];
			visible_board[local_y][local_x]=local_temp1;
		}
	}
	visible_board_white_pieces=new_board_white_pieces;
	visible_board_black_pieces=new_board_black_pieces;
#endif
}

bool show_winner()
{
	if (new_board_black_pieces==0)
		return true;
	return false;
}


byte new_game()
{
	bool move_done;
	bool turns_skipped;

	// New Game
	init_first_board();
	copy_new_board_to_visible();

	turns_skipped=0;
	move_done=false;

	for(;;)
	{
		find_possible_moves_and_count_pieces();
		copy_new_board_to_visible();
		show_board();
		show_score();

		if (game_turn!=1)
			mark_last_move();

		if (game_turn==MAX_TURNS+1)
			break;

		if (new_board_black_pieces==0 || new_board_white_pieces==0)
			break;

		if (turns_skipped==2) // both players have no move
			break;

		if (new_board_possible_moves_number>0)
		{
			turns_skipped=0;

			if ((current_color==BLACK && black_computer_controlled) || (current_color==WHITE && white_computer_controlled))
				move_done=get_ai_action();
			else
			{
				move_done=get_player_action();
			}

			if (move_done)
			{
				make_move(current_color);
				++game_turn; 
				last_x=move_x;
				last_y=move_y;
				last_color=current_color;
			}
			else
			{
				return EMPTY;
			}
		}
		else
		{
			++turns_skipped;
		}
		switch_sides();
	}

	if (new_board_black_pieces>new_board_white_pieces)
	{
		return BLACK;
	}
	else if (new_board_black_pieces<new_board_white_pieces)
	{
		return WHITE;
	}
	return EMPTY;
}

#if __ATARI__
void main(void)
{
	system_init();

	black_computer_controlled=false;
	white_computer_controlled=true;
	current_color=BLACK;
	tree_depth=1; // 2,4,6

	for (;;)
	{
		configure_game();
		winner(new_game());
	}
}
#endif